/* $Id: set.c,v 1.23 1999/10/04 23:42:38 donwm Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* This file is used to get or set parameters. */

#include "sema.h"

SHORTSIZ16
i1432_get_chan_float(E1432ID hw, SHORTSIZ16 ID,
		     LONGSIZ32 reg, FLOATSIZ32 * ptr)
{
    SHORTSIZ16 error;
    LONGSIZ32 ltmp;

    TRACE_PRINTF(1, ("i1432_get_chan_float(0x%p, %d, %ld, 0x%p)\n",
		     hw, ID, reg, ptr));

    error = i1432_get_chan_long(hw, ID, reg, &ltmp);
    if (error)
	return error;
    *ptr = *(FLOATSIZ32 *) &ltmp;

    return 0;
}

SHORTSIZ16
i1432_get_chan_long(E1432ID hw, SHORTSIZ16 ID,
		    LONGSIZ32 reg, LONGSIZ32 * ptr)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    SHORTSIZ16 error;
    LONGSIZ32 temp;
    int     first, cindex;

    TRACE_PRINTF(1, ("i1432_get_chan_long(0x%p, %ld, %ld, 0x%p)\n",
		     hw, ID, reg, ptr));

    /* Check for valid id */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    /* Check if channel or group */
    if (ID < 0)
    {
	/* Iterate thru group making sure that all parameters match */
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;

	/* Check each channel */
	first = 1;
	while (cn)
	{
	    error = i1432_get_chan_long(hw, cn->chanID, reg, &temp);
	    if (error)
		return error;
	    if (first)
	    {
		*ptr = temp;
		first = 0;
	    }
	    if (temp != *ptr)
		return i1432_print_error(ERR1432_PARAMETER_UNEQUAL);
	    cn = cn->next;
	}
    }
    else
    {
	/* get lone channel */
	cindex = i1432_get_chan_index(hw, ID);
	return e1432_read32_register(hw, ID, reg + cindex *
				     E1432_CREG_OFFSET, ptr);
    }

    return 0;
}

SHORTSIZ16
i1432_get_chan_short(E1432ID hw, SHORTSIZ16 ID,
		     LONGSIZ32 reg, SHORTSIZ16 * ptr)
{
    SHORTSIZ16 error;
    LONGSIZ32 ltmp;

    TRACE_PRINTF(1, ("i1432_get_chan_short(0x%p, %ld, %ld, 0x%p)\n",
		     hw, ID, reg, ptr));

    error = i1432_get_chan_long(hw, ID, reg, &ltmp);
    if (error)
	return error;
    *ptr = (SHORTSIZ16) ltmp;

    return 0;
}

SHORTSIZ16
i1432_get_chan_limits(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 parm_index,
		      FLOATSIZ32 *min, FLOATSIZ32 *max,
		      FLOATSIZ32 *def, FLOATSIZ32 *step)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    SHORTSIZ16 error;
    FLOATSIZ32 tmin, tmax, tdef, tstep;
    int     first;

    TRACE_PRINTF(1, ("i1432_get_chan_limits(0x%p, %d, %ld, "
		     "0x%p, 0x%p, 0x%p, 0x%p)\n",
		     hw, ID, parm_index, min, max, def, step));

    /* Check for valid id */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    /* check if channel or group */
    if (ID < 0)
    {
	/* iterate thru group */
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;

	first = 1;
	while (cn)
	{
	    error = i1432_get_chan_limits(hw, cn->chanID, parm_index, &tmin,
					  &tmax, &tdef, &tstep);
	    if (error)
		return error;
	    if (first)
	    {
		*min = tmin;
		*max = tmax;
		*def = tdef;
		*step = tstep;
		first = 0;
	    }
	    /* Floating-point compare, sort of dangerous!!! */
	    if (tmin != *min ||
		tmax != *max ||
		tdef != *def ||
		tstep != *step)
		return i1432_print_error(ERR1432_PARAMETER_UNEQUAL);
	    cn = cn->next;
	}
    }
    else
    {
	int sca = i1432_get_chan_sca(hw, ID);
	LONGSIZ32 limits = E1432_LIM_SCA
	  + (sca * E1432_LIM_SCA_OFFSET + parm_index) * E1432_LIM_ENTRIES;

	if ( ! error )
	{
	    error = e1432_read32_register(hw, ID,
	      limits + E1432_LIM_MIN * sizeof (FLOATSIZ32), (LONGSIZ32 *)min);
	}
	if ( ! error )
	{
	    error = e1432_read32_register(hw, ID,
	      limits + E1432_LIM_MAX * sizeof (FLOATSIZ32), (LONGSIZ32 *)max);
	}
	if ( ! error )
	{
	    error = e1432_read32_register(hw, ID,
	      limits + E1432_LIM_DEF * sizeof (FLOATSIZ32), (LONGSIZ32 *)def);
	}
	if ( ! error )
	{
	    error = e1432_read32_register(hw, ID,
	      limits + E1432_LIM_STEP * sizeof (FLOATSIZ32), (LONGSIZ32 *)step);
	}
    }

    return error;
}

SHORTSIZ16
i1432_set_chan(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 reg,
	       LONGSIZ32 cmd, LONGSIZ32 value)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    SHORTSIZ16 error, state;
    LONGSIZ32 tmp;
    int     cindex, type;

    TRACE_PRINTF(1, ("i1432_set_chan(0x%p, %d, %ld, %ld, %ld)\n",
		     hw, ID, reg, cmd, value));

    /* Check for valid id */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    /* check if channel or group */
    if (ID < 0)
    {
	/* iterate thru group */
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;
	while (cn)
	{
	    error = i1432_set_chan(hw, cn->chanID, reg, cmd, value);
	    if (error)
		return error;
	    cn = cn->next;
	}
	return 0;
    }
    else
    {
	/* set lone channel */
	cindex = i1432_get_chan_index(hw, ID);

	if (reg != -1)
	{
	    /* Getting a parameter value is fast compared to setting
	       one, so first check to see if the new value will be
	       identical to the old one.  If so, then don't bother
	       setting the parameter. */
	    error = e1432_read32_register(hw, ID, reg + cindex *
					  E1432_CREG_OFFSET, &tmp);
	    if (error)
		return error;
	    if (tmp == value)
		return 0;
	}

	error = i1432_write_cmd2(hw, ID, cmd, cindex, value);
	if (error)
	    return error;

	/* When some parameters are changed during a measurement, they
	   change the scale factors used to scale input data.  We need
	   to update these scale factors by calling
	   e1432_cached_parm_update with the appropriate IDs */
	type = (ID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	if (type == E1432_CHAN_TYPE_INPUT)
	    switch (cmd)
	    {
	    case E1432_CCMD_SET_ANTI_ALIAS_DIGITAL:
	    case E1432_CCMD_SET_INPUT_HIGH:
	    case E1432_CCMD_SET_RANGE:
		error = e1432_get_meas_state(hw, ID, &state);
		if (error)
		    return error;
		if (state > E1432_MEAS_STATE_TESTED)
		    /* Just update this channel's scale factors */
		    return e1432_cached_parm_update(hw, ID);
	    }
	else if (type == E1432_CHAN_TYPE_TACH)
	    switch (cmd)
	    {
	    case E1432_CCMD_SET_INPUT_HIGH:
	    case E1432_CCMD_SET_TRIGGER_LEVEL_LOWER:
	    case E1432_CCMD_SET_TRIGGER_LEVEL_UPPER:
	    case E1432_CCMD_SET_TRIGGER_SLOPE:
		error = e1432_get_meas_state(hw, ID, &state);
		if (error)
		    return error;
		if (state > E1432_MEAS_STATE_TESTED)
		    /* Update the scale factors for all input channels in
		       the system.  Not efficient, but worth a try... */
		    for (cindex = 1;
			 cindex <= i1432_chan_count[E1432_CHAN_TYPE_INPUT];
			 cindex++)
		    {
			error = e1432_cached_parm_update
			    (hw, (SHORTSIZ16)((E1432_CHAN_TYPE_INPUT <<
					       E1432_CHAN_TYPE_SHIFT)
					      + cindex));
			if (error)
			    return error;
		    }
		break;
	    }

	return 0;
    }
}

SHORTSIZ16
i1432_get_mod_float(E1432ID hw, SHORTSIZ16 ID,
		    LONGSIZ32 reg, FLOATSIZ32 * ptr)
{
    SHORTSIZ16 error;
    LONGSIZ32 ltmp;

    TRACE_PRINTF(1, ("i1432_get_mod_short(0x%p, %ld, %ld, 0x%p)\n",
		     hw, ID, reg, ptr));

    error = i1432_get_mod_long(hw, ID, reg, &ltmp);
    if (error)
	return error;
    *ptr = *(FLOATSIZ32 *) &ltmp;

    return 0;
}

SHORTSIZ16
i1432_get_mod_long(E1432ID hw, SHORTSIZ16 ID,
		   LONGSIZ32 reg, LONGSIZ32 * ptr)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error;
    LONGSIZ32 temp;
    int     first, mod;

    TRACE_PRINTF(1, ("i1432_get_mod_long(0x%p, %ld, %ld, 0x%p)\n",
		     hw, ID, reg, ptr));

    /* Check for valid id */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    /* Check if channel or group */
    if (ID < 0)
    {
	/* Iterate through group making sure that all parameters match */
	first = 1;
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    /* Do one module */
	    mn = gn->modlist[mod];
	    error =
		i1432_get_mod_long(hw,
				   i1432_get_chan_from_module(mn),
				   reg, &temp);
	    if (error)
		return error;
	    if (first)
	    {
		*ptr = temp;
		first = 0;
	    }
	    if (temp != *ptr)
		return i1432_print_error(ERR1432_PARAMETER_UNEQUAL);
	}
    }
    else
	/* get lone module */
	return e1432_read32_register(hw, ID, reg, ptr);

    return 0;
}

SHORTSIZ16
i1432_get_mod_short(E1432ID hw, SHORTSIZ16 ID,
		    LONGSIZ32 reg, SHORTSIZ16 * ptr)
{
    SHORTSIZ16 error;
    LONGSIZ32 ltmp;

    TRACE_PRINTF(1, ("i1432_get_mod_short(0x%p, %ld, %ld, 0x%p)\n",
		     hw, ID, reg, ptr));

    error = i1432_get_mod_long(hw, ID, reg, &ltmp);
    if (error)
	return error;
    *ptr = (SHORTSIZ16) ltmp;

    return 0;
}

SHORTSIZ16
i1432_get_mod_limits(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 cmd,
		     FLOATSIZ32 *min, FLOATSIZ32 *max,
		     FLOATSIZ32 *def, FLOATSIZ32 *step)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error, error2;
    FLOATSIZ32 tmin, tmax, tdef, tstep;
    int     first, mod, cindex;

    TRACE_PRINTF(1, ("i1432_get_mod_limits(0x%p, %d, %ld, "
		     "0x%p, 0x%p, 0x%p, 0x%p)\n",
		     hw, ID, cmd, min, max, def, step));

    /* Check for valid id */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    /* Check if channel or group */
    if (ID < 0)
    {
	/* Iterate through group making sure that all parameters match */
	first = 1;
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    /* Do one module */
	    mn = gn->modlist[mod];
	    error = i1432_get_mod_limits(hw,
					 i1432_get_chan_from_module(mn),
					 cmd, &tmin, &tmax, &tdef,
					 &tstep);
	    if (error)
		return error;
	    if (first)
	    {
		*min = tmin;
		*max = tmax;
		*def = tdef;
		*step = tstep;
		first = 0;
	    }
	    /* Floating-point compare, sort of dangerous!!! */
	    if (tmin != *min ||
		tmax != *max ||
		tdef != *def ||
		tstep != *step)
		return i1432_print_error(ERR1432_PARAMETER_UNEQUAL);
	}
    }
    else
    {
	/* get lone module */
	cindex = i1432_get_chan_index(hw, ID);

	error = i1432_introff();
	if (error)
	    return error;

	error = i1432_write_cmd1(hw, ID, cmd, cindex);
	if (error)
	    goto cleanup;
	error = i1432_read_resp4(hw, ID, (LONGSIZ32 *) min,
				 (LONGSIZ32 *) max,
				 (LONGSIZ32 *) def,
				 (LONGSIZ32 *) step);
	if (error)
	    goto cleanup;

    cleanup:
	error2 = i1432_intron();
	if (error2)
	    return error2;

	if (error)
	    return error;
    }

    return 0;
}

SHORTSIZ16
i1432_set_mod(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 reg,
	      LONGSIZ32 cmd, LONGSIZ32 value)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error;
    LONGSIZ32 tmp;
    int     mod;

    TRACE_PRINTF(1, ("i1432_set_mod(0x%p, %d, %ld, %ld, %ld)\n",
		     hw, ID, reg, cmd, value));

    /* Check for valid id */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    /* Check if channel or group */
    if (ID < 0)
    {
	/* Iterate through group */
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    /* Do one module */
	    mn = gn->modlist[mod];
	    error = i1432_set_mod(hw,
				  i1432_get_chan_from_module(mn),
				  reg, cmd, value);
	    if (error)
		return error;
	}
	return 0;
    }
    else
    {
	/* Set lone module */
	if (reg != -1)
	{
	    /* Getting a parameter value is fast compared to setting
	       one, so first check to see if the new value will be
	       identical to the old one.  If so, then don't bother
	       setting the parameter. */
	    error = e1432_read32_register(hw, ID, reg, &tmp);
	    if (error)
		return error;
	    if (tmp == value)
		return 0;
	}

	return i1432_write_cmd1(hw, ID, cmd, value);
    }
}


SHORTSIZ16
i1432_set_mod_nowait(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 reg,
	      LONGSIZ32 cmd, LONGSIZ32 value)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error, error2, mod_id;
    LONGSIZ32 tmp;
    int     mod;

    TRACE_PRINTF(1, ("i1432_set_mod(0x%p, %d, %ld, %ld, %ld)\n",
		     hw, ID, reg, cmd, value));

    /* Check for valid id */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    /* Check if channel or group */
    if (ID < 0)
    {
	error = i1432_introff();
	if (error)
	return error;

	/* Iterate through group */
	gn = i1432_get_group_node(hw, ID);
	/* send parmater and command, don't wait */
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    /* Do one module */
	    mn = gn->modlist[mod];
	    mod_id = i1432_get_chan_from_module(mn);
	    if (reg != -1)
	    {
		/* Getting a parameter value is fast compared to setting
		   one, so first check to see if the new value will be
		   identical to the old one.  If so, then don't bother
		   setting the parameter. */
		error = e1432_read32_register(hw, mod_id, reg, &tmp);
		if (error)
		    goto cleanup;
		if (tmp == value)
		    goto cleanup;
	    }
	    error = i1432_write_cmd1_start(hw, mod_id, cmd, value);
	    if (error)
		goto cleanup;
	}
	/* now wait for command to finish */
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    /* Do one module */
	    mn = gn->modlist[mod];
	    mod_id = i1432_get_chan_from_module(mn);
	    error = i1432_write_cmd_finish(hw, mod_id, cmd);
	    if (error)
		goto cleanup;
	}
    }
    else
    {
	/* Set lone module */
	/* since only one module, no need to do separate command finish */
	if (reg != -1)
	{
	    /* Getting a parameter value is fast compared to setting
	       one, so first check to see if the new value will be
	       identical to the old one.  If so, then don't bother
	       setting the parameter. */
	    error = e1432_read32_register(hw, ID, reg, &tmp);
	    if (error)
		return error;
	    if (tmp == value)
		return 0;
	}

	return i1432_write_cmd1(hw, ID, cmd, value);
    }

 cleanup:
    error2 = i1432_intron();
    if (error2)
	return error2;

    if (error)
	return error;

    return 0;
}


